import { world, system, EntityComponentTypes, ItemComponentTypes, TicksPerSecond, Dimension, Block } from '@minecraft/server';
import { Vec3 } from '../utils/Vec3';
import { Helper } from '../utils/Helper';
import { CompostablesPlusItems } from './CompostablesPlusItems';
class CompostablesPlus {
  static COMPOSTER_TYPE = 'minecraft:composter';
  static COMPOSTER_FILL_LEVEL_STATE = 'composter_fill_level';
  constructor() {
    this.compostRegisteredCompostables();
    this.compostByDropping();
  }
  getRegisteredCompostableChance(itemType) {
    return CompostablesPlusItems.find(compostable => itemType === compostable.item)?.chance;
  }
  getComposterFillLevel(composter) {
    return composter.permutation.getState(CompostablesPlus.COMPOSTER_FILL_LEVEL_STATE);
  }
  isFillableComposter(composter) {
    return this.getComposterFillLevel(composter) < 7;
  }
  composting(dimension, composter, compostingChance) {
    const composterFillLevel = this.getComposterFillLevel(composter);
    const composterCenter = composter.center();
    const { COMPOSTER_FILL_LEVEL_STATE } = CompostablesPlus;
    if (Math.random() < compostingChance / 100 || composterFillLevel <= 0) {
      const newComposterFillLevel = composterFillLevel + 1;
      composter.setPermutation(composter.permutation.withState(COMPOSTER_FILL_LEVEL_STATE, newComposterFillLevel));
      if (newComposterFillLevel === 7) system.runTimeout(() => {
        if (!composter?.isValid() || this.getComposterFillLevel(composter) != newComposterFillLevel) return;
        composter.setPermutation(composter.permutation.withState(COMPOSTER_FILL_LEVEL_STATE, 8));
        dimension.playSound('block.composter.ready', composterCenter);
      }, TicksPerSecond);
      dimension.playSound('block.composter.fill_success', composterCenter);
    } else dimension.playSound('block.composter.fill', composterCenter);
    dimension.playSound('item.bone_meal.use', composterCenter, { pitch: Math.random() * (1.1 - 0.9) + 0.9 });
    dimension.spawnParticle('minecraft:crop_growth_emitter', composterCenter);
  }
  compostRegisteredCompostables() {
    world.beforeEvents.itemUseOn.subscribe(({ block, itemStack, source: player }) => {
      if (player.isSneaking || block.typeId !== CompostablesPlus.COMPOSTER_TYPE || !this.isFillableComposter(block)) return;
      const compostingChance = this.getRegisteredCompostableChance(itemStack.typeId);
      if (!compostingChance) return;
      system.run(() => {
        try {
          this.composting(player.dimension, block, compostingChance);
          Helper.decreaseMainhandItemStack(player);
        } catch {}
      });
    });
  }
  compostByDropping() {
    world.afterEvents.entitySpawn.subscribe(({ entity }) => {
      if (entity?.typeId !== 'minecraft:item') return;
      const itemStack = entity.getComponent(EntityComponentTypes.Item)?.itemStack;
      const compostingChance = itemStack?.getComponent(ItemComponentTypes.Compostable)?.compostingChance || this.getRegisteredCompostableChance(itemStack?.typeId);
      if (!compostingChance) return;
      const fallingItem = system.runInterval(() => {
        if (!entity?.isValid()) {
          system.clearRun(fallingItem);
          return;
        }
        if (entity?.isFalling) return;
        const { dimension, location } = entity;
        const verifyLoc = Vec3.fromMinecraftVector3(location);
        verifyLoc.y -= location.y > Math.floor(location.y) + 0.5 ? 0.5 : 0.1;
        const block = dimension.getBlock(verifyLoc);
        if (block?.typeId !== CompostablesPlus.COMPOSTER_TYPE) {
          system.clearRun(fallingItem);
          return;
        }
        if (!this.isFillableComposter(block)) return;
        let stackAmount = itemStack.amount;
        for (; stackAmount > 0 && this.isFillableComposter(block); stackAmount--) {
          if (stackAmount > 1) --itemStack.amount;
          this.composting(dimension, block, compostingChance);
        }
        if (stackAmount > 0) dimension.spawnItem(itemStack, location);
        entity.remove();
        system.clearRun(fallingItem);
      }, 5);
    });
  }
}
new CompostablesPlus();